function res = denton_uni(Y,x,ta,d,sc)
% PURPOSE: Temporal disaggregation using the Denton method
% -----------------------------------------------------------------------
% SYNTAX: res = denton_uni(Y,x,ta,d,sc);
% -----------------------------------------------------------------------
% OUTPUT: res: a structure
%         res.meth  = 'Denton';
%         res.N     = Number of low frequency data
%         res.ta    = Type of disaggregation
%         res.sc     = Frequency conversion
%         res.d     = Degree of differencing 
%         res.y     = High frequency estimate
%         res.x     = High frequency indicator
%         res.U     = Low frequency residuals
%         res.u     = High frequency residuals
%         res.et    = Elapsed time
% -----------------------------------------------------------------------
% INPUT: Y: Nx1 ---> vector of low frequency data
%        x: nx1 ---> vector of low frequency data
%        ta: type of disaggregation
%            ta=1 ---> sum (flow)
%            ta=2 ---> average (index)
%            ta=3 ---> last element (stock) ---> interpolation
%            ta=4 ---> first element (stock) ---> interpolation
%        d: objective function to be minimized: volatility of ...
%            d=0 ---> levels
%            d=1 ---> first differences
%            d=2 ---> second differences
%        sc: number of high frequency data points for each low frequency data point 
%            sc= 4 ---> annual to quarterly
%            sc=12 ---> annual to monthly
%            sc= 3 ---> quarterly to monthly
% -----------------------------------------------------------------------
% LIBRARY: aggreg, bfl
% -----------------------------------------------------------------------
% SEE ALSO: tduni_plot, tduni_print
% -----------------------------------------------------------------------
% REFERENCE: Denton, F.T. (1971) "Adjustment of monthly or quarterly 
% series to annual totals: an approach based on quadratic minimization", 
% Journal of the American Statistical Society, vol. 66, n. 333, p. 99-102.

% written by:
%  Enrique M. Quilis
%  Macroeconomic Research Department
%  Ministry of Economy and Finance
%  Paseo de la Castellana, 162. Office 2.5-1.
%  28046 - Madrid (SPAIN)
%  <enrique.quilis@meh.es>

% Version 1.1 [August 2006]

t0=clock;

% -----------------------------------------------------------------------
% Size of the problem

[N,M] = size(Y);
[n,m] = size(x);

if ((m > 1) | (M > 1))
    error ('*** INCORRECT DIMENSIONS ***');
else
    clear m M;
end

% -----------------------------------------------------------------------
% Generation of aggregation matrix C

C = aggreg(ta,N,sc);

% -----------------------------------------------------------
% Expanding the aggregation matrix to perform
% extrapolation if needed.

if (n > sc * N)
   pred = n - sc*N;           % Number of required extrapolations 
   C = [C zeros(N,pred)];
else
   pred = 0;
end

% -----------------------------------------------------------------------
% Temporal aggregation matrix of the indicator

X = C*x;

% -----------------------------------------------------------------------
% Computation of low frequency residuals

U = Y - X;

% -----------------------------------------------------------------------
% Computation of high frequency series = indicator + annual residuals
% temporally disaggregated by means of Boot-Feibes-Lisman

res_bfl = bfl(U,ta,d,sc);
u = res_bfl.y;

% Expand residuals if needed (extrapolation case)
if (pred > 0)
   u = [u ; zeros(pred,1)];
end

y = x + u;

% -----------------------------------------------------------------------
% Loading the structure
% -----------------------------------------------------------------------

% Basic parameters 

res.meth = 'Denton';
res.N 	= N;
res.ta	= ta;
res.sc 	= sc;
res.pred	= pred;
res.d 	= d;

% -----------------------------------------------------------------------
% Series

res.y = y;
res.x = x;
res.U = U;
res.u = u;

% -----------------------------------------------------------------------
% Elapsed time

res.et  = etime(clock,t0);
