function []=td_plot(res)
% PURPOSE: Generate graphic output of temporal disaggregation methods
% ------------------------------------------------------------
% SYNTAX: td_plot(res);
% ------------------------------------------------------------
% OUTPUT: Detailed grahic output:
%         - Objective function vs. rho
%         - Estimate and one-sigma limits
%         - High frequency conformity: estimate and indicators (levels)
%         - Low frequency series vs low frequency indicators (scaled)
%         - High frequency conformity: yoy rates of growth
%         - High frequency conformity: yoy rates of growth (last periods)
%         - High and low frequency series: y vs Y
%         - High and low frequency residuals: u vs U
% ------------------------------------------------------------
% INPUT: res: structure generated by td programs
% ------------------------------------------------------------
% LIBRARY: rate, impz, aggreg, sacf, spacf
% ------------------------------------------------------------
% SEE ALSO: chowlin, fernandez, litterman, td_plot

% written by:
%  Enrique M. Quilis
%  Macroeconomic Research Department
%  Ministry of Economy and Finance
%  Paseo de la Castellana, 162. Office 2.5-1.
%  28046 - Madrid (SPAIN)
%  <enrique.quilis@meh.es>

% Version 1.1 [August 2006]

% -----------------------------------------------------------
% -----------------------------------------------------------
% Loading the structure

meth=res.meth;

% -----------------------------------------------------------
% Basic parameters 
N=res.N;
n=res.n;
pred=res.pred;
sc=res.sc;
p=res.p;
ta=res.ta;
type=res.type;

% -----------------------------------------------------------
% Series
Y = res.Y;
x = res.x;
y    = res.y;
d_y  = res.y_dt;
y_li = res.y_lo;
y_ls = res.y_up;

% -----------------------------------------------------------
% Residuals
u = res.u;
U = res.U;

% -----------------------------------------------------------
% Parameters
beta = res.beta;
beta_sd =res.beta_sd;
beta_t =res.beta_t;
rho = res.rho;

% -----------------------------------------------------------
% Information criteria
aic=res.aic;
bic=res.bic;

% -----------------------------------------------------------
% Selection of periodicity of high frequency data
% Low-frequency (lf) and high-frequency (hf) depends on the
% problem at hand. The default options are related to sc 
% according to:
%                       sc
%  :::::::::::::::::::::::::::::::::::::::::::::::::::::::
%               3                  4                12
%  :::::::::::::::::::::::::::::::::::::::::::::::::::::::
%  lf =         4                  1                 1
%  hf =        12                  4 = sc            12 = sc
%  :::::::::::::::::::::::::::::::::::::::::::::::::::::::
%

if (sc==3)
    lf = 4;
    hf = 12;
else
    lf = 1;
    hf = sc;
end 

% -----------------------------------------------------------
% -----------------------------------------------------------
% Graphic output

% -----------------------------------------------------------
% Objective function vs. rho

switch res.meth
case {'Fernandez'}
   % Do nothing
case {'Chow-Lin','Litterman','Santos Silva-Cardoso'}
    switch type
        case {0,1}
            figure;
            plot(res.r,res.wls,'ro',res.r,res.wls);
            title  ('Weighted least squares vs innovational/dynamic parameter');
            ylabel ('weighted least squares'); xlabel ('innovational/dynamic parameter');
            grid on;
            figure;
            plot(res.r,res.loglik,'ro',res.r,res.loglik);
            title  ('Likelihood vs innovational/dynamic parameter');
            ylabel ('likelihood'); xlabel ('innovational/dynamic parameter');
            grid on;
        case 3 % Chow-Lin by Cochrane-Orcutt
            plot(res.r,res.val,res.ropt,ones(1,length(res.ropt)),'b', ...
                res.r,res.valA,'ro',res.ropt,res.r,'ro');
            title  ('Cochrane-Orcutt. Annual vs quarterly rho');
            ylabel ('Annual rho'); xlabel ('innovational/dynamic parameter');
            figure;
            subplot(1,2,1); 
            plot(res.co.iter(:,3),res.co.iter(:,1),'ro',res.co.iter(:,3),res.co.iter(:,1),'b-'); 
            axis([0 res.co.iter(end) 0 1]); 
            xlabel('Iteration'); ylabel('Annual rho'); title ('Cochrane-Orcutt estimation');
            subplot(1,2,2); 
            plot(res.co.iter(:,3),res.co.iter(:,2),'ro',res.co.iter(:,3),res.co.iter(:,2),'b-'); 
            axis([0 res.co.iter(end) 0 1.05*max(res.co.iter(:,2))]); 
            xlabel('Iteration'); ylabel('Convergence'); title ('Cochrane-Orcutt estimation');
            grid on;
        case 4
            % Just do nothing
    end; 
end

% -----------------------------------------------------------
% MA expansion of (1-phi B) filter of SSC method

switch res.meth
case {'Santos Silva-Cardoso'}
   figure;
   impz([1],[1 -res.rho]);
   title ('IMPULSE-RESPONSE FUNCTION OF (1-phi*B) FILTER')
end

% -----------------------------------------------------------
% Estimate and one-sigma limits

figure;
t=1:n;
plot(t,y,'r-',t,y_li,'b-',t,y_ls,'b-');
grid off; legend ('y','y-sigma','y+sigma',0);
title ('High frequency estimate +/- sigma ');
xlabel ('time');

% -----------------------------------------------------------------
% High frequency conformity: estimate and scaled indicator (levels)

figure;
xb=x*beta;
t=1:n;
plot(t,y,'r-',t,xb,'b-');
grid off; legend ('y','x*beta',0);
title ('High frequency conformity');
xlabel ('time');

% --------------------------------------------------------------
% Only if there is only one indicator (without intercept)
if (p == 2)
   % -----------------------------------------------------------
   % High frequency conformity: yoy rates of growth
   figure;
   ty=rate(y,hf);  % Computing yoy rates of growth
   tx=rate(x(:,2),hf);
   t=1:n;
   plot(t,ty,'b-*',t,tx,'r-');
   grid off; legend ('y','x',0);
   title ('High frequency conformity: yoy rates of growth');
   xlabel ('time');
   % -----------------------------------------------------------
   % High frequency conformity: yoy rates of growth
   % of last 3*sc data
   figure;
   ty_li=rate(y_li,hf);
   ty_ls=rate(y_ls,hf);
   t=n-3*sc:n;
   plot(t,ty(t),'r-o',t,ty_ls(t),'r:',t,ty_li(t),'r:',t,tx(t),'b-');
   grid off; legend ('y','y+sigma','y-sigma','x',0);
   title ('High frequency conformity: yoy rates of growth (detail)');
   xlabel ('time');
end;   

% -----------------------------------------------------------
% Low frequency series vs low frequency indicators (scaled): levels

C=aggreg(ta,N,sc); 
X=C*x(1:n-pred,:);  % Only low freq. data span is considered

figure;
t=1:N;
plot(t,Y,'ro-',t,X*beta,'b+-');
grid off; legend ('Y','X*beta',0);
title ('Low frequency conformity');
xlabel ('time'); 

% -----------------------------------------------------------
% Only if there is only one indicator (without intercept)

if (p == 2) 
    % -----------------------------------------------------------
    % Low frequency series vs low frequency indicators: yoy growth
    figure;
    t=1:N;
    plot(t,rate(Y,lf),'ro-',t,rate(X(:,2),lf),'b+-');
    grid off; legend ('Y','X',0);
    title ('Low frequency conformity: yoy rates of growth');
    xlabel ('time'); 
end
   
% -----------------------------------------------------------
% High and low frequency series: y vs Y
%   1. Generate a high freq. time series with the low freq. span
%   2. Plot both series (low freq. divided by g1)

if (ta == 1)
   g1=sc;
else
   g1=1;
end

i=1; t=1;
while (t<=N)   
   c=0;
   while (c<sc)
      ya(i,1)=Y(t,1);
      c=c+1;
      i=i+1;
   end
   t=t+1;
end;

figure;
t=1:n-pred;
y_N=y(1:n-pred);  % Only the part of low freq. span is considered
plot(t,y_N,'r-',t,ya/g1,'b-');
grid off; legend (meth,'y naive',0);
title ('High frequency and low frequency series');
xlabel ('time'); 

% -----------------------------------------------------------
% High and low frequency residuals: u vs U
%   1. Generate a high freq. time series with the low freq. data
%   2. Plot both series (low freq. divides by g1)

i=1; t=1;
while (t<=N)
   c=0;
   while (c<sc)
      ua(i,1)=U(t,1);
      c=c+1;
      i=i+1;
   end
   t=t+1;
end;

figure;
t=1:n-pred;
u_N=u(1:n-pred);   % Only the part of low freq. data
plot(t,u_N,'r-',t,ua/g1,'b-');
grid off; legend ('u','U',0);
title ('High frequency and low frequency residuals');
xlabel ('time'); 

% ACF and PACF of low-frequency residuals
figure;
subplot(1,2,1); sacf(res.U,res.sc,0);
subplot(1,2,2); spacf(res.U,res.sc);
legend ('LF residuals');

% ACF and PACF of high-frequency residuals
figure;
subplot(1,2,1); sacf(res.u,3*res.sc,0);
subplot(1,2,2); spacf(res.u,3*res.sc);
legend ('hf residuals');
